<?php
/**
 * Admin Interface Class
 * Handles the admin menu and tabbed interface
 */

if (!defined('ABSPATH')) {
    exit;
}

class GBX_Admin {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }
    
    public function add_admin_menu() {
        add_menu_page(
            'GBX Blog Suite',
            'GBX Blog Suite',
            'manage_options',
            'gbx-blog-suite',
            array($this, 'render_admin_page'),
            'dashicons-admin-site-alt3',
            30
        );
    }
    
    public function enqueue_scripts($hook) {
        if ($hook !== 'toplevel_page_gbx-blog-suite') {
            return;
        }
        
        wp_enqueue_style('gbx-admin-style', GBX_BLOG_SUITE_URL . 'assets/admin-style.css', array(), GBX_BLOG_SUITE_VERSION);
    }
    
    public function render_admin_page() {
        $active_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'api-publisher';
        ?>
        <div class="wrap">
            <h1>GBX Blog Suite</h1>
            
            <h2 class="nav-tab-wrapper">
                <a href="?page=gbx-blog-suite&tab=api-publisher" 
                   class="nav-tab <?php echo $active_tab === 'api-publisher' ? 'nav-tab-active' : ''; ?>">
                    API Publisher
                </a>
                <a href="?page=gbx-blog-suite&tab=internal-links" 
                   class="nav-tab <?php echo $active_tab === 'internal-links' ? 'nav-tab-active' : ''; ?>">
                    Internal Links
                </a>
                <a href="?page=gbx-blog-suite&tab=blog-settings" 
                   class="nav-tab <?php echo $active_tab === 'blog-settings' ? 'nav-tab-active' : ''; ?>">
                    Blog Settings
                </a>
            </h2>
            
            <div class="gbx-tab-content">
                <?php
                switch ($active_tab) {
                    case 'api-publisher':
                        $this->render_api_publisher_tab();
                        break;
                    case 'internal-links':
                        $this->render_internal_links_tab();
                        break;
                    case 'blog-settings':
                        $this->render_blog_settings_tab();
                        break;
                }
                ?>
            </div>
        </div>
        <?php
    }
    
    private function render_api_publisher_tab() {
        $api_publisher = gbx_blog_suite()->api_publisher;
        $daily_enabled = get_option('rpp_daily_enabled', false);
        $publish_status = get_option('rpp_publish_status', 'publish');
        ?>
        <div class="gbx-api-publisher-tab gbx-card">
            <h2>Manual Publishing</h2>
            <p class="description">Click the button below to fetch and publish a post from the API immediately.</p>
            <div class="gbx-action-box">
                <button id="rpp-fetch-btn" class="button button-primary button-hero">Fetch & Publish Post</button>
            </div>
            <div id="rpp-result"></div>

            <hr class="gbx-divider">

            <h2>Daily Auto-Publishing</h2>
            <p class="description">Enable daily automatic publishing to fetch and publish one post every day.</p>
            
            <table class="gbx-status-table">
                <tr>
                    <td><strong>Status:</strong></td>
                    <td>
                        <?php if ($daily_enabled): ?>
                            <span class="gbx-badge gbx-badge-success">Enabled</span>
                        <?php else: ?>
                            <span class="gbx-badge gbx-badge-disabled">Disabled</span>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <td><strong>Post Status:</strong></td>
                    <td>
                        <select id="rpp-status-select">
                            <option value="draft" <?php selected($publish_status, 'draft'); ?>>Draft</option>
                            <option value="publish" <?php selected($publish_status, 'publish'); ?>>Publish</option>
                        </select>
                    </td>
                </tr>
            </table>

            <div class="gbx-action-box">
                <?php if ($daily_enabled): ?>
                    <button id="rpp-disable-schedule" class="button button-large">Disable Daily Schedule</button>
                <?php else: ?>
                    <button id="rpp-enable-schedule" class="button button-primary button-large">Enable Daily Schedule</button>
                <?php endif; ?>
            </div>
            <div id="rpp-schedule-result"></div>
            
            <hr class="gbx-divider">
            
            <h2>Integration</h2>
            <div class="gbx-integration-box">
                <span class="dashicons dashicons-yes-alt"></span> <strong>All in One SEO</strong> is supported
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            $('#rpp-fetch-btn').click(function() {
                var btn = $(this);
                btn.prop('disabled', true).text('Publishing...');
                $('#rpp-result').html('<p style="color:#666;">Fetching data from API...</p>');
                
                $.post(ajaxurl, {
                    action: 'rpp_fetch_post',
                    nonce: '<?php echo wp_create_nonce('rpp_fetch_nonce'); ?>'
                }, function(response) {
                    if (response.success) {
                        $('#rpp-result').html('<div class="notice notice-success inline"><p>' + response.data + '</p></div>');
                    } else {
                        $('#rpp-result').html('<div class="notice notice-error inline"><p>' + response.data + '</p></div>');
                    }
                    btn.prop('disabled', false).text('Fetch & Publish Post');
                });
            });
            
            $('#rpp-status-select').change(function() {
                $.post(ajaxurl, {
                    action: 'rpp_update_status',
                    status: $(this).val(),
                    nonce: '<?php echo wp_create_nonce('rpp_status_nonce'); ?>'
                });
            });
            
            $('#rpp-enable-schedule').click(function() {
                var btn = $(this);
                btn.prop('disabled', true);
                
                $.post(ajaxurl, {
                    action: 'rpp_enable_schedule',
                    nonce: '<?php echo wp_create_nonce('rpp_schedule_nonce'); ?>'
                }, function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        $('#rpp-schedule-result').html('<div class="notice notice-error inline"><p>' + response.data + '</p></div>');
                        btn.prop('disabled', false);
                    }
                });
            });
            
            $('#rpp-disable-schedule').click(function() {
                var btn = $(this);
                btn.prop('disabled', true);
                
                $.post(ajaxurl, {
                    action: 'rpp_disable_schedule',
                    nonce: '<?php echo wp_create_nonce('rpp_schedule_nonce'); ?>'
                }, function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        $('#rpp-schedule-result').html('<div class="notice notice-error inline"><p>' + response.data + '</p></div>');
                        btn.prop('disabled', false);
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    private function render_internal_links_tab() {
        ?>
        <div class="gbx-internal-links-tab gbx-card">
            <h2><span class="dashicons dashicons-admin-links"></span> Smart Internal Links</h2>
            <div class="gbx-content-pad">
                <p class="gbx-intro">The Smart Internal Links feature analyzes your content while you edit posts to suggest relevant internal linking opportunities.</p>
                
                <div class="gbx-step-box">
                    <h3>How to Use:</h3>
                    <ol class="gbx-steps">
                        <li>Go to <strong>Posts → All Posts</strong> and edit any post.</li>
                        <li>Scroll down to find the <strong>"Smart Internal Links"</strong> meta box.</li>
                        <li>Click the <strong>"Analyze Content"</strong> button.</li>
                        <li>Review suggestions and click <strong>"Add Link"</strong> to insert them instantly.</li>
                    </ol>
                </div>

                <div class="gbx-action-box">
                    <a href="<?php echo admin_url('edit.php'); ?>" class="button button-primary button-large">Go to Posts & Start Linking</a>
                </div>
            </div>
        </div>
        <?php
    }
    
    private function render_blog_settings_tab() {
        // Get current settings
        $display_name = get_option('gbx_display_name', '');
        $local_avatar = get_option('gbx_local_avatar', '');
        $cf7_recaptcha_enabled = get_option('gbx_cf7_recaptcha_enabled', 0);
        $author_display_enabled = get_option('gbx_author_display_enabled', 0);
        ?>
        <div class="gbx-blog-settings-tab gbx-card">
            <form id="gbx-blog-settings-form" method="post">
                <?php wp_nonce_field('gbx_blog_settings_nonce', 'gbx_settings_nonce'); ?>
                
                <h2>General Settings</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="gbx_display_name">Display Name</label>
                        </th>
                        <td>
                            <input type="text" id="gbx_display_name" name="display_name" value="<?php echo esc_attr($display_name); ?>" class="regular-text" />
                            <p class="description">Set your display name (updates your user profile).</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="gbx_local_avatar">Local Avatar</label>
                        </th>
                        <td>
                            <div class="gbx-avatar-wrapper">
                                <input type="text" id="gbx_local_avatar" name="local_avatar" value="<?php echo esc_url($local_avatar); ?>" class="regular-text" readonly />
                                <button type="button" class="button" id="gbx_upload_avatar">Select Image</button>
                                <button type="button" class="button button-link-delete" id="gbx_remove_avatar" <?php echo empty($local_avatar) ? 'style="display:none;"' : ''; ?>>Remove</button>
                            </div>
                            <?php if (!empty($local_avatar)): ?>
                                <div id="gbx_avatar_preview" class="gbx-preview-box">
                                    <img src="<?php echo esc_url($local_avatar); ?>" />
                                </div>
                            <?php endif; ?>
                            <p class="description">Upload a custom avatar image (saved to your user meta).</p>
                        </td>
                    </tr>
                </table>
                
                <hr class="gbx-divider">

                <h2>Integrations</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="gbx_cf7_recaptcha_enabled">CF7 reCAPTCHA v2</label>
                        </th>
                        <td>
                            <select id="gbx_cf7_recaptcha_enabled" name="cf7_recaptcha_enabled">
                                <option value="0" <?php selected($cf7_recaptcha_enabled, 0); ?>>Default</option>
                                <option value="1" <?php selected($cf7_recaptcha_enabled, 1); ?>>reCAPTCHA v2</option>
                            </select>
                            <p class="description">Enable reCAPTCHA v2 integration for Contact Form 7.</p>
                        </td>
                    </tr>
                </table>

                <hr class="gbx-divider">
                
                <h2>Display Rules</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="gbx_author_display_enabled">Author Information</label>
                        </th>
                        <td>
                            <label class="gbx-toggle">
                                <input type="checkbox" id="gbx_author_display_enabled" name="author_display_enabled" value="1" <?php checked($author_display_enabled, 1); ?> />
                                <span class="gbx-toggle-label">Hide all author information</span>
                            </label>
                            <p class="description">Completely remove author metadata from frontend pages.</p>
                        </td>
                    </tr>
                </table>
                
                <div class="gbx-footer-actions">
                    <button type="submit" class="button button-primary button-large" id="gbx_save_settings">Save Settings</button>
                    <span id="gbx_settings_message" class="gbx-message"></span>
                </div>
            </form>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // Avatar upload
            var avatarFrame;
            $('#gbx_upload_avatar').click(function(e) {
                e.preventDefault();
                
                if (avatarFrame) {
                    avatarFrame.open();
                    return;
                }
                
                avatarFrame = wp.media({
                    title: 'Select Avatar Image',
                    button: { text: 'Use this image' },
                    multiple: false
                });
                
                avatarFrame.on('select', function() {
                    var attachment = avatarFrame.state().get('selection').first().toJSON();
                    $('#gbx_local_avatar').val(attachment.url);
                    $('#gbx_remove_avatar').show();
                    
                    if ($('#gbx_avatar_preview').length) {
                        $('#gbx_avatar_preview img').attr('src', attachment.url);
                    } else {
                        $('#gbx_remove_avatar').after('<div id="gbx_avatar_preview" style="margin-top: 10px;"><img src="' + attachment.url + '" style="max-width: 100px; height: auto; border: 1px solid #ddd; padding: 5px;" /></div>');
                    }
                });
                
                avatarFrame.open();
            });
            
            $('#gbx_remove_avatar').click(function() {
                $('#gbx_local_avatar').val('');
                $('#gbx_avatar_preview').remove();
                $(this).hide();
            });
            
            // Save settings
            $('#gbx-blog-settings-form').submit(function(e) {
                e.preventDefault();
                
                var $btn = $('#gbx_save_settings');
                var $msg = $('#gbx_settings_message');
                
                $btn.prop('disabled', true).text('Saving...');
                $msg.html('');
                
                $.post(ajaxurl, {
                    action: 'gbx_save_blog_settings',
                    nonce: $('#gbx_settings_nonce').val(),
                    display_name: $('#gbx_display_name').val(),
                    local_avatar: $('#gbx_local_avatar').val(),
                    cf7_recaptcha_enabled: $('#gbx_cf7_recaptcha_enabled').val(),
                    author_display_enabled: $('#gbx_author_display_enabled').is(':checked') ? 1 : 0
                }, function(response) {
                    $btn.prop('disabled', false).text('Save Settings');
                    
                    if (response.success) {
                        $msg.html('<span style="color: green;">✓ ' + response.data + '</span>');
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        $msg.html('<span style="color: red;">✗ ' + response.data + '</span>');
                    }
                });
            });
        });
        </script>
        <?php
    }
}
