<?php
/**
 * Blog Settings Class
 * Handles blog customization settings
 */

if (!defined('ABSPATH')) {
    exit;
}

class GBX_Blog_Settings {
    
    public function __construct() {
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp_ajax_gbx_save_blog_settings', array($this, 'save_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        
        // Apply CF7 reCAPTCHA v2 if enabled
        add_action('plugins_loaded', array($this, 'init_cf7_recaptcha'));
        
        // Apply author display rule
        add_action('init', array($this, 'apply_author_display_rule'));
    }
    
    public function register_settings() {
        register_setting('gbx_blog_settings', 'gbx_display_name');
        register_setting('gbx_blog_settings', 'gbx_local_avatar');
        register_setting('gbx_blog_settings', 'gbx_cf7_recaptcha_enabled');
        register_setting('gbx_blog_settings', 'gbx_author_display_enabled');
    }
    
    public function enqueue_scripts($hook) {
        if ($hook === 'toplevel_page_gbx-blog-suite') {
            wp_enqueue_media();
        }
    }
    
    public function save_settings() {
        check_ajax_referer('gbx_blog_settings_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $messages = array();
        
        // 1. Save Display Name
        if (isset($_POST['display_name'])) {
            $display_name = sanitize_text_field($_POST['display_name']);
            if (!empty($display_name)) {
                $current_user_id = get_current_user_id();
                $update_user = wp_update_user(array(
                    'ID' => $current_user_id,
                    'display_name' => $display_name,
                    'nickname' => $display_name
                ));
                if (!is_wp_error($update_user)) {
                    update_option('gbx_display_name', $display_name); // Keep redundancy for display
                    $messages[] = 'Display name updated.';
                }
            }
        }
        
        // 3. Save Local Avatar
        if (isset($_POST['local_avatar'])) {
            $local_avatar = esc_url_raw($_POST['local_avatar']);
            update_option('gbx_local_avatar', $local_avatar);
            $current_user_id = get_current_user_id();
            update_user_meta($current_user_id, 'gbx_local_avatar', $local_avatar);
        }
        
        // 4. Save CF7 reCAPTCHA v2 setting
        if (isset($_POST['cf7_recaptcha_enabled'])) {
            $cf7_enabled = intval($_POST['cf7_recaptcha_enabled']);
            update_option('gbx_cf7_recaptcha_enabled', $cf7_enabled);
        }
        
        // 5. Save Author Display Rule
        if (isset($_POST['author_display_enabled'])) {
            $author_display = intval($_POST['author_display_enabled']);
            update_option('gbx_author_display_enabled', $author_display);
        }
        
        wp_send_json_success('Settings saved successfully!');
    }
    
    public function init_cf7_recaptcha() {
        if (!get_option('gbx_cf7_recaptcha_enabled', 0)) {
            return;
        }
        
        // Load the internal reCAPTCHA v2 implementation
        $recaptcha_file = GBX_BLOG_SUITE_PATH . 'includes/class-cf7-recaptcha-v2.php';
        
        if (file_exists($recaptcha_file) && class_exists('WPCF7')) {
            if (!class_exists('GBX_CF7_Recaptcha_V2')) {
                require_once $recaptcha_file;
            }
        }
    }
    
    public function apply_author_display_rule() {
        if (!get_option('gbx_author_display_enabled', 0)) {
            return;
        }
        
        // Load the no-author-anywhere functionality
        $no_author_file = GBX_BLOG_SUITE_PATH . 'no-author-anywhere/no-author-anywhere/no-author-anywhere.php';
        if (file_exists($no_author_file)) {
            if (!class_exists('NoAuthorAnywhere')) {
                include_once $no_author_file;
            }
        }
    }
}
